<?php
/**
 * API communication class
 */

if (!defined('ABSPATH')) {
    exit;
}

class WP_SSL_API {
    
    private $api_url;
    private $client_id;
    private $client_secret;
    
    public function __construct() {
        $this->api_url = get_option('wp_ssl_api_url', '');
        $this->client_id = get_option('wp_ssl_client_id', '');
        $this->client_secret = get_option('wp_ssl_client_secret', '');
    }
    
    public function create_certificate($domain, $san_domains = array(), $verification_method = 'http-01', $provider = 'letsencrypt') {
        $response = wp_remote_post($this->api_url . '/api/v1/certificates', array(
            'headers' => array(
                'X-Client-Id' => $this->client_id,
                'X-Client-Secret' => $this->client_secret,
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'domain' => $domain,
                'san_domains' => $san_domains,
                'verification_method' => $verification_method,
                'provider' => $provider,
            )),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 201 && isset($body['success']) && $body['success']) {
            return $body;
        } else {
            return array(
                'success' => false,
                'message' => isset($body['message']) ? $body['message'] : 'Failed to create certificate',
            );
        }
    }
    
    public function get_verification($certificate_id, $verification_id) {
        $response = wp_remote_get($this->api_url . '/api/v1/certificates/' . $certificate_id . '/verifications', array(
            'headers' => array(
                'X-Client-Id' => $this->client_id,
                'X-Client-Secret' => $this->client_secret,
            ),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            return null;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['success']) && $body['success'] && isset($body['data'])) {
            foreach ($body['data'] as $verification) {
                if ($verification['id'] == $verification_id) {
                    return $verification;
                }
            }
        }
        
        return null;
    }
    
    public function verify_domain($certificate_id, $verification_id) {
        $response = wp_remote_post($this->api_url . '/api/v1/certificates/' . $certificate_id . '/verify', array(
            'headers' => array(
                'X-Client-Id' => $this->client_id,
                'X-Client-Secret' => $this->client_secret,
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'verification_id' => $verification_id,
            )),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 200 && isset($body['success']) && $body['success']) {
            return $body;
        } else {
            return array(
                'success' => false,
                'message' => isset($body['message']) ? $body['message'] : 'Verification failed',
            );
        }
    }
    
    public function download_certificate($certificate_id, $format = 'nginx') {
        $response = wp_remote_get($this->api_url . '/api/v1/certificates/' . $certificate_id . '/download?format=' . $format, array(
            'headers' => array(
                'X-Client-Id' => $this->client_id,
                'X-Client-Secret' => $this->client_secret,
            ),
            'timeout' => 60,
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 200 && isset($body['success']) && $body['success']) {
            return $body;
        } else {
            return array(
                'success' => false,
                'message' => isset($body['message']) ? $body['message'] : 'Download failed',
            );
        }
    }
    
    public function get_certificates($status = null) {
        $url = $this->api_url . '/api/v1/certificates';
        if ($status) {
            $url .= '?status=' . urlencode($status);
        }
        
        $response = wp_remote_get($url, array(
            'headers' => array(
                'X-Client-Id' => $this->client_id,
                'X-Client-Secret' => $this->client_secret,
            ),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 200 && isset($body['success']) && $body['success']) {
            return $body;
        } else {
            return array(
                'success' => false,
                'message' => isset($body['message']) ? $body['message'] : 'Failed to fetch certificates',
            );
        }
    }
    
    public function get_certificate($certificate_id) {
        $response = wp_remote_get($this->api_url . '/api/v1/certificates/' . $certificate_id, array(
            'headers' => array(
                'X-Client-Id' => $this->client_id,
                'X-Client-Secret' => $this->client_secret,
            ),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 200 && isset($body['success']) && $body['success']) {
            return $body;
        } else {
            return array(
                'success' => false,
                'message' => isset($body['message']) ? $body['message'] : 'Failed to fetch certificate',
            );
        }
    }
}

